'------------------------------------------------------------------------------
' This program is Copyright (c) 2005 Microsoft Corporation.
'
' All rights reserved.
'
' THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
' ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
' THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
' PARTICULAR PURPOSE.
'
' IN NO EVENT SHALL MICROSOFT AND/OR ITS RESPECTIVE SUPPLIERS BE
' LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
' DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
' WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
' ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
' OF THIS CODE OR INFORMATION.
'------------------------------------------------------------------------------
'
'     ISA Server 2004 (Objets COM FPC)
'
'     Log records extraction corresponding to a given string (URL or Rule)
'     Result is a texte file - v1.8
'
'     Concept : Query ISA logs to retrieve records matching string and time
'               criterias then writes results in a time ascending order texte
'               file respecting fields order if /f argument is specified
'
'               ISA API capability doesn't exceed 20000 records for a query
'               This script workaround this by splitting the time period in
'               many log queries(/i).
'
'     Arguments : /u : String to search as firewall rule records
'                      (If http://xxx format, search as URL record)
'                 /All : All records (equiv *)
'                 /Full : Records all ISA available fields (default)
'                 /r : Name of report file
'                 /m : Max Nb of records (def = 20000)
'                 /i : Query interval (in minutes)
'                 /start : Start date for the query (def = J-7)
'                 /end : End date (def = J)
'                 /LastDay : Logs for J-1
'                 /LastWeek : Logs for last week, starting on Sunday
'                 /d : First day of the week (0 = Sunday, 1 = Monday)
'                 /LastMonth : Logs for past month, starting the 1st
'                 /Last30 : 30 past days
'                 /Last24 : 24 past hours
'                 /Last1 : Last hour
'                 /Last7 : Last 7 days
'                 /List : List of available fields in ISA Logs
'                 /Load : Load wanted fields from a file
'                 /f : Name of the file used with /List or /Load
'                 /s : Fields separator (default : " ; ")
'
'                 /h or /? or /help : Display help
'
'     Microsoft Consulting Services - jcornier@microsoft.com - 2005/01
'------------------------------------------------------------------------------

Option Explicit

Dim objFPCRoot  ' FPCLib.FPC root object for ISA Server 2004
Dim iErrorNumber
Dim strErrorDescription

Dim bDisplay, bAllRecords, bList
Dim bLast1, bLast7, bLast24, bLast30, bStartDate, bEndDate
Dim bLastDay, bLastWeek, bLastMonth

dim strSeparator
Dim ArrFields()

' Create FPC (ISA) object
Set objFPCRoot = CreateObject("FPC.Root")

bDisplay = TRUE

Main

'------------------------------------------------------------------------------
'    Procdure Principale
'------------------------------------------------------------------------------
Sub Main()
    Dim objArgs
    Dim iArgs, iNbArgs
    Dim bURLok
    Dim strCurrDir, strFileName, strQuestion, strLogListFile
    Dim iLoop, iPosHttp, iWeekDayToStart, iNbLogs
    Dim strLogString
    Dim StartProcess
    Dim oFile
    Dim TxtCache
    Dim strUrlRoot, strUrlSource, strUrlName
    Dim strStartDate, strEndDate, strWeekDayToStart, strYearMonth
    Dim iRecords, iQueryInterval
    Dim StartDate, EndDate, StopDate
    Dim strLogRecord
    Dim strQueryInterval
    Dim QueryInterval

    ' Default values
    iRecords = 20000     ' Don't set more than 20000 ! (ISA constraint)
    iQueryInterval = 15  ' In minutes, from 2 to 60
    iNbLogs = 0          ' Max Nb of records to retrieve in one query
    strLogListFile = "loglist.txt"
    strSeparator = " ; "
    bAllRecords = TRUE
    bList = FALSE
    bLast7 = FALSE
    bLast1 = FALSE
    bLast24 = FALSE
    bLast30 = FALSE
    bStartDate = FALSE
    bEndDate = FALSE
    bLastDay = FALSE
    bLastWeek = FALSE
    bLastMonth = FALSE

    Set objArgs = WScript.Arguments
    iNbArgs = (objArgs.Count - 1)

    ' Set current path for files (strCurrDir)
    '--------------------------------------------------------------------------
    Set oFile = WScript.CreateObject("Scripting.FileSystemObject")
    strCurrDir = oFile.GetAbsolutePathName(".")
    Set oFile = Nothing

    '--------------------------------------------------------------------------
    ' Set values given in command line mode
    '--------------------------------------------------------------------------
    If iNbArgs >= 0 Then

        bDisplay = FALSE
        bURLok = FALSE

        For iArgs = 0 to iNbArgs
            Select Case LCase(objArgs(iArgs))
                Case "/?", "-h", "/help", "-help" ' Help
                    DisplayHelp
                Case "/u", "-u" ' Url or Rule to search
                    If (iArgs + 1) <= iNbArgs Then
                        strUrlRoot = objArgs(iArgs + 1)
                        bURLok = TRUE
                    Else
                        call DisplayHelp
                    End If
                Case "/full", "-full" ' all available fields
                    bAllRecords = TRUE
                Case "/all", "-all" ' all rules
                    strUrlRoot = "*"
                    bURLok = TRUE
                Case "/r", "-r" ' Name of report file
                    If (iArgs + 1) <= iNbArgs Then
                        strFileName = objArgs(iArgs + 1)
                    Else
                        call DisplayHelp
                    End If
                Case "/m", "-m" ' Max Records
                    If (iArgs + 1) <= iNbArgs Then
                        iRecords = CInt(objArgs(iArgs + 1))
                    Else
                        call DisplayHelp
                    End If
                Case "/i", "-i" ' Query interval (in minutes)
                    If (iArgs + 1) <= iNbArgs Then
                        iQueryInterval = CInt(objArgs(iArgs + 1))
                    Else
                        call DisplayHelp
                    End If
                Case "/start", "-start" ' Start Date
                    If (iArgs + 1) <= iNbArgs Then
                        bStartDate = TRUE
                        StartDate = DateValue(objArgs(iArgs + 1))
                    Else
                        call DisplayHelp
                    End If
                Case "/end", "-end" ' End Date
                    If (iArgs + 1) <= iNbArgs Then
                        bEndDate = TRUE
                        EndDate = DateValue(objArgs(iArgs + 1))
                    Else
                        call DisplayHelp
                    End If
                Case "/d", "-d" ' First day of the week
                    If (iArgs + 1) <= iNbArgs Then
                        iWeekDayToStart = WeekDayNumber(objArgs(iArgs + 1))
                    Else
                        call DisplayHelp
                    End If
                Case "/lastday", "-lastday" ' J - 1
                    bLastDay = TRUE
                Case "/lastweek", "-lastweek" ' Last Week, start Sunday
                    bLastWeek = TRUE
                Case "/lastmonth", "-lastmonth" ' Last Month
                    bLastMonth = TRUE
                Case "/last1", "-last1" ' Last hour
                    bLast1 = TRUE
                Case "/last24", "-last24" ' Last 24h
                    bLast24 = TRUE
                Case "/last7", "-last7" ' Last 7 days
                    bLast7 = TRUE
                Case "/last30", "-last30" ' Last 30 days
                    bLast30 = TRUE
                Case "/list", "-list" ' List of available fields
                    bList = TRUE
                Case "/load", "-load" ' Load wanted fields
                    bAllRecords = FALSE
                Case "/f", "-f" ' Name of file used with /list or /load
                                ' default : "loglist.txt"
                    If (iArgs + 1) <= iNbArgs Then
                        strLogListFile = objArgs(iArgs + 1)
                    Else
                        call DisplayHelp
                    End If
                Case "/s", "-s" ' Fields separator (default " ; ")
                    If (iArgs + 1) <= iNbArgs Then
                        strSeparator = objArgs(iArgs + 1)
                    Else
                        call DisplayHelp
                    End If
            End Select
        Next

        If bList Then
            ' Warning : create file with list of available fields, then QUIT
            call ListLogFields(strLogListFile)
        End If

        If bURLok = FALSE Then
            call DisplayHelp
        End If

    Else
    '--------------------------------------------------------------------------
    ' If no arguments detected in command line, then use message boxes
    '--------------------------------------------------------------------------
        '______________________________________________________________________
        ' Request for a list of available fields
        ' or load to specify the file with wanted fields,
        ' or "full" to retrieve all fields (default : full)
        strLogListFile = InputBox ("Report file: " & VbCrLf & _
            "- Full to retrieve all avilable fields" & VbCrLf & _
            "- List to create a file with available fields" & VbCrLf & _
            "- Load to specify a file with wanted fields" & VbCrLf & VbCrLf & _
            "(This can run in command line mode - " & _
            "Option /h to list arguments)", _
            "Report file", "full")
        If strLogListFile = "" Then
            call ProcessCanceled("Script canceled by user")
        End If
        If LCase(strLogListFile) = "full" Then
            bAllRecords = TRUE
        End If

        If LCase(strLogListFile) = "list" Then
            ' Name of file used to list all fields
            strLogListFile = InputBox ("Name of list file: " & VbCrLf & _
                "All available fields will be listed in file.", _
                "List file name", _
                strCurrDir & "\" & "loglist.txt")
                If strLogListFile = "" Then
                    call ProcessCanceled("Script canceled by user")
                End If

            ' Warning : create file, then QUIT
            call ListLogFields(strLogListFile)
        End If

        If LCase(strLogListFile) = "load" Then
            ' Name of file used to specify fields
            strLogListFile = InputBox ("Name of list file: " & VbCrLf & _
                "Ordered list of fields to retrieve", _
                "List file name", _
                strCurrDir & "\" & "loglist.txt")
                If strLogListFile = "" Then
                    call ProcessCanceled("Script canceled by user")
                End If
            bAllRecords = FALSE
        End If
        '______________________________________________________________________

        ' Search criteria
        strUrlRoot = InputBox ("String to search for in logs: " & _
            VbCrLf & VbCrLf & _
            "Firewall rules are used as search source." & VbCrLf & _
            "* retrieve all records." & VbCrLf & VbCrLf & _
            "An http://xxx format search on URL instead of FW Rules.", _
            "Search criteria", "*")
        If strUrlRoot = "" Then
            call ProcessCanceled("Script canceled by user")
        End If

        ' Start date
        strStartDate = InputBox ("Start date, " & _
            "values :" & VbCrLf & _
            "   LastDay - yesterday" & VbCrLf & _
            "   Last1 - last hour" & VbCrLf & _
            "   Last24 - last 24h" & VbCrLf & _
            "   Last7 - last 7 days" & VbCrLf & _
            "   Last30 - last 30 days" & VbCrLf & _
            "   LastWeek - last week" & VbCrLf & _
            "   LastMonth - last month starting the 1st", _
            "Start date", "LastDay")
        Select Case LCase(strStartDate)
            Case ""
                call ProcessCanceled("Script canceled by user")
            Case "lastday"
                bLastDay = TRUE
            Case "lastweek"
                bLastWeek = TRUE
            Case "lastmonth"
                bLastMonth = TRUE
            Case "last1"
                bLast1 = TRUE
            Case "last24"
                bLast24 = TRUE
            Case "last7"
                bLast7 = TRUE
            Case "last30"
                bLast30 = TRUE
            Case Else
                StartDate = DateValue(strStartDate)
                bStartDate = TRUE
        End Select

        ' End date
        If bStartDate Then
            strEndDate = InputBox ("End date: " & VbCrLf & _
                "(None to specify now)", _
                "End date", "")
            Select Case LCase(strEndDate)
                Case ""
                    call ProcessCanceled("Script canceled by user")
                Case "None"
                    bEndDate = FALSE
                Case Else
                    EndDate = DateValue(strEndDate)
                    bEndDate = TRUE
            End Select
        End If

        ' First day of the week
        If bLastWeek Then
            strWeekDayToStart = InputBox ("First day of the week: " & VbCrLf & _
                "(Values: Sunday, Monday, ...)", _
                "First week day", "Sunday")
            If WeekDayNumber(strWeekDayToStart) = -1 Then
                call ProcessCanceled("Script canceled by user")
            End If
        End If

        ' Extract usable string from search string to build report file name
        ' ex: http://www.site.com/test -> www.site.com
        strFileName = strUrlRoot
        iPosHttp = InStr(strFileName, "http://")
        If iPosHttp > 0 Then
            strFileName = Mid(strFileName, 8, Len(strFileName) - 7)
            iPosHttp = InStr(strFileName, "/")
            If iPosHttp > 0 Then
                strFileName = Left(strFileName, iPosHttp - 1)
            End If
        End If
        If InStr(strFileName, "*") Then
            strFileName = "All_Records"
        End If
        If bLastDay Then
            strFileName = strDate(Date() - 1) & "_" & strFileName & ".txt"
        ElseIf bLastMonth Then
            Select Case Month(Date()) - 1
                Case 0
                     strYearMonth = Year(Date()) - 1 & "12"
                Case 1, 2, 3, 4, 5, 6, 7, 8, 9
                     strYearMonth = Year(Date()) & "0" & Month(Date()) - 1
                Case Else
                     strYearMonth = Year(Date()) & Month(Date()) - 1
            End Select
            strFileName = strYearMonth & "_" & strFileName & ".txt"
        ElseIf bLast1 Then
            strFileName = strDate(Date()) & "_" & Hour(Now()) & Minute(Now()) & "_" & strFileName & ".txt"
        Else
            strFileName = strDate(Date()) & "_" & strFileName & ".txt"
        End If
        strFileName = InputBox ("Report file name: ", _
            "Report file name", _
            strFileName)
        If strFileName = "" Then
            call ProcessCanceled("Script canceled by user, " & _
                VbCrLf & "or no file name was given.")
        End If

        ' Max. Nb of records for one ISA log query
        iRecords = CInt(InputBox ("Max Nb of records for one ISA log query:" & _
            VbCrLf & " (Max value is : 20000)", _
            "Max Nb of records", iRecords))
        If iRecords = 0 Then
            call ProcessCanceled("Script canceled by user")
        End If

        ' Query interval in minutes
        ' Max 1h (value 60), Min 1 minute
        iQueryInterval = CInt(InputBox ("Query interval: " & VbCrLf & _
            "Max nb of minutes for one query" & VbCrLf & _
            "(in minutes, min 1, max 60):", _
            "Query interval" , iQueryInterval))
        If iQueryInterval = 0 Then
            call ProcessCanceled("Script canceled by user")
        End If

    End If

    '--------------------------------------------------------------------------
    ' start of automatic process
    '--------------------------------------------------------------------------
    StartProcess = Date()
    
    ' Control values, calculate start and end dates
    '--------------------------------------------------------------------------
    If iQueryInterval < 1 Then
        QueryInterval = TimeValue("00:01:00")
    ElseIf iQueryInterval >= 60 Then
        QueryInterval = TimeValue("01:00:00")
    ElseIf iQueryInterval < 10 Then
        QueryInterval = TimeValue("00:0" & iQueryInterval & ":00")
    Else
        QueryInterval = TimeValue("00:" & iQueryInterval & ":00")
    End If

    If bStartDate Then
        bLast1 = FALSE
        bLast7 = FALSE
        bLast24 = FALSE
        bLast30 = FALSE
        bLastDay = FALSE
        bLastWeek = FALSE
        bLastMonth = FALSE
    End If

    If bLast1 Then
        bLast7 = FALSE
        bLast24 = FALSE
        bLast30 = FALSE
        bLastDay = FALSE
        bLastWeek = FALSE
        bLastMonth = FALSE
        bStartDate = TRUE         ' FALSE to work in regular mode
        bEndDate = TRUE           ' FALSE to work in regular mode
        ' Now - 1 hour doesn't give the right result,
        ' solution is : now - 1 day + 23 hours :-)
        StartDate = (Now() - 1) + TimeValue("23:00:00")
        EndDate = Now()
    End If

    If bLast24 Then
        bLast1 = FALSE
        bLast7 = FALSE
        bLast30 = FALSE
        bLastDay = FALSE
        bLastWeek = FALSE
        bLastMonth = FALSE
        bStartDate = TRUE         ' FALSE to work in regular mode
        bEndDate = TRUE           ' FALSE to work in regular mode
        StartDate = Now() - 1
        EndDate = Now()
    End If

    If bLast7 Then
        bLast1 = FALSE
        bLast24 = FALSE
        bLast30 = FALSE
        bLastDay = FALSE
        bLastWeek = FALSE
        bLastMonth = FALSE
        bStartDate = TRUE         ' FALSE to work in regular mode
        bEndDate = TRUE           ' FALSE to work in regular mode
        StartDate = Date() - 7
        EndDate = Date()
    End If

    If bLast30 Then
        bLast1 = FALSE
        bLast7 = FALSE
        bLast24 = FALSE
        bLastDay = FALSE
        bLastWeek = FALSE
        bLastMonth = FALSE
        bStartDate = TRUE         ' FALSE to work in regular mode
        bEndDate = TRUE           ' FALSE to work in regular mode
        StartDate = Date() - 31
        EndDate = Date()
    End If

    If bLastDay Then
        bLast1 = FALSE
        bLast7 = FALSE
        bLast24 = FALSE
        bLast30 = FALSE
        bLastWeek = FALSE
        bLastMonth = FALSE
        bStartDate = TRUE
        bEndDate = TRUE
        StartDate = Date() - 1
        EndDate = Date()
    End If

    If bLastWeek Then
        bLast1 = FALSE
        bLast7 = FALSE
        bLast24 = FALSE
        bLast30 = FALSE
        bLastDay = FALSE
        bLastMonth = FALSE
        bStartDate = TRUE
        bEndDate = TRUE
        ' iWeekDayToStart = 0 (default) starts the week on sunday at midnight
        ' and stop next sunday midnight too.
        ' To start on monday for example, give 1 as argument
        StartDate = Date() - 7 - WeekDay(Date()) + 1 + iWeekDayToStart
        EndDate = Date() - 7 - WeekDay(Date()) + 8 + iWeekDayToStart
        ' If end date is after today, then starts on sunday
        If EndDate > Date() Then
            StartDate = Date() - 7 - WeekDay(Date()) + 1
            EndDate = Date() - 7 - WeekDay(Date()) + 8
        End If
    End If

    If bLastMonth Then
        bLast1 = FALSE
        bLast7 = FALSE
        bLast24 = FALSE
        bLast30 = FALSE
        bLastDay = FALSE
        bLastWeek = FALSE
        bStartDate = TRUE
        bEndDate = TRUE
        ' this Month -1, end the first day of this month at midnight
        StartDate = Date() - Day(Date()) - Day(Date() - Day(Date())) + 1
        EndDate = Date() - Day(Date()) + 1
    End If

    If Not (bStartDate And bEndDate) Then
        ' Default if lastDay (J-1)
        bLastDay = TRUE
        bStartDate = TRUE
        bEndDate = TRUE
        StartDate = Date() - 1
        EndDate = Date()
    End If

    '--------------------------------------------------------------------------
    ' Preserved code for debug purpopse
    ' Control of time periods used for queries

    'WScript.Echo "bLast1 " &  bLast1 & VbCrLf & _
    '    "bLast7 " & bLast7 & VbCrLf & _
    '    "bLast24 " & bLast24 & VbCrLf & _
    '    "bLast30 " & bLast30 & VbCrLf & _
    '    "bLastDay " & bLastDay & VbCrLf & _
    '    "bLastWeek " & bLastWeek & VbCrLf & _
    '    "bLastMonth " & bLastMonth & VbCrLf & _
    '    "bStartDate " & bStartDate & VbCrLf & _
    '    "bEndDate " & bEndDate & VbCrLf & _
    '    "bAllRecords " & bAllRecords & VbCrLf & _
    '    VbCrLf & _
    '    "StartDate " & StartDate & VbCrLf & _
    '    "EndDate "  & EndDate
    'WScript.Quit(0)
    
    '--------------------------------------------------------------------------
    ' Report file name control
    ' Set full path report file name (erase if exist)
    ' Current path is choose if none was given as argument
    '--------------------------------------------------------------------------
    If strFileName = "" or InStr(strFileName, "*") Then
        iPosHttp = InStr(strUrlRoot, "http://")
        If iPosHttp > 0 Then
            strFileName = Mid(strUrlRoot, 8, Len(strUrlRoot) - 7)
            iPosHttp = InStr(strFileName, "/")
            If iPosHttp > 0 Then
                strFileName = Left(strFileName, iPosHttp - 1)
            End If
        Else
            strFileName = strUrlRoot
        End If
        If InStr(strUrlRoot, "*") Then
            strFileName = "All_Records"
        End If
        If bLastDay Then
            strFileName = strDate(Date() - 1) & "_" & strFileName & ".txt"
        ElseIf bLastMonth Then
            Select Case Month(Date()) - 1
                Case 0
                     strYearMonth = Year(Date()) - 1 & "12"
                Case 1, 2, 3, 4, 5, 6, 7, 8, 9
                     strYearMonth = Year(Date()) & "0" & Month(Date()) - 1
                Case Else
                     strYearMonth = Year(Date()) & Month(Date()) - 1
            End Select
            strFileName = strYearMonth & "_" & strFileName & ".txt"
        ElseIf bLast1 Then
            strFileName = strDate(Date()) & "_" & Hour(Now()) & Minute(Now()) & "_" & strFileName & ".txt"
        Else
            strFileName = strDate(Date()) & "_" & strFileName & ".txt"
        End If
    End If

    strFileName = strCurrDir & "\" & strFileName
    
    '--------------------------------------------------------------------------
    ' Write first line with choosen fields
    '--------------------------------------------------------------------------
    If bAllRecords Then
        call LoadFields("None")
    Else
        call LoadFields(strLogListFile)
    End If
    
    For iLoop = 1 To UBound(ArrFields)
        If iLoop = 1 Then
            strLogRecord = "#" & ArrFields(iLoop)
        Else
            strLogRecord = strLogRecord & strSeparator & ArrFields(iLoop)
        End If
    Next

    call LogAction(strFileName, strLogRecord, TRUE)
    '__________________________________________________________________________
    '______________________________ Core ______________________________________

    '--------------------------------------------------------------------------
    ' Split of the search time period in many intervals of X minutes
    ' to stay into the limit of 20000 records per log query
    '--------------------------------------------------------------------------
    StopDate = EndDate
    EndDate = StartDate + QueryInterval

    Do While DateDiff("s", EndDate, StopDate) >= 0
    
        ' Debug purpose, next two line give the current query interval
        '----------------------------------------------------------------------
        'strLogString = StartDate & " ; " & EndDate & " ; " & StopDate
        'call LogAction(strFileName, strLogString, FALSE)

        '----------------------------------------------------------------------
        ' Search in logs for matching objects
        '----------------------------------------------------------------------
        call SiteLogs(objFPCRoot, strUrlRoot, strFileName, iRecords, _
            StartDate, EndDate, iNbLogs)
            
        ' Debug purpose, add a line separator between sets of records
        'call LogAction(strFileName, "              -----              ", FALSE)

        StartDate = EndDate
        EndDate = EndDate + QueryInterval
        If DateDiff("s", EndDate, StopDate) <= 0 Then
            ' Last query
            EndDate = StopDate
            If StartDate = EndDate Then
                ' Set EndDate to exit current loop
                EndDate = EndDate + 1
            End IF
        End If
    Loop

    '_____________________________ End Core ___________________________________
    '__________________________________________________________________________

    ' Append a line at enf of report
    'strLogString = VbCrLf & Date() & " " & Time()
    'WScript.Echo strLogString
    'call LogAction(strFileName, strLogString, FALSE)

    If bDisplay Then
        ' Display Query information if needed
        WScript.Echo "Racine : " & strUrlRoot & _
            VbCrLf & "Nb Records Max : " & iRecords & _
            VbCrLf & "Nb Records " & iNbLogs
        'WScript.Echo "End"
    End If
End Sub

'------------------------------------------------------------------------------
'                         End of main code
'                   Start of procedure and functions
'------------------------------------------------------------------------------

'----------------------------------------------------------------
'    Purpose : Display a message,
'             then quit the script on user request
'    Arguments : Message to display in message box
'----------------------------------------------------------------
Sub ProcessCanceled(StrQuestion)

    MsgBox strQuestion,vbExclamation + vbOkOnly + vbApplicationModal + 256, _
        "Fin du Script"
    WScript.Quit(1)
End Sub

'----------------------------------------------------------------
'    Purpose : Build a date string
'    Return : YearMonthDay, format "yyyymmdd"
'----------------------------------------------------------------
Function strDate(dDate)
    Dim strToday, strMonth, strDay, strYear

    strToday = dDate
    strMonth = Month(strToday)
    strDay = Day(strToday)
    If Len(strMonth) < 2 Then strMonth = "0" & strMonth End If
    If Len(strDay) < 2 Then strDay = "0" & strDay End If
    strDate = Year(strToday) & strMonth & strDay
End Function

'----------------------------------------------------------------
'    Purpose : Display help
'----------------------------------------------------------------
Sub DisplayHelp()

    WScript.Echo "ISA Server 2004 - Log Extraction for an URL or FW Rule" & _
    VbCrLf & _
    VbCrLf & "     Usage:" & _
    VbCrLf & "          /u: String to search as firewall rule records" & _
    VbCrLf & "              (If http://xxx format, search as URL record)" & _
    VbCrLf & _
    VbCrLf & "         Optional:" & _
    VbCrLf & "          /Full: Records all ISA available fields (default)" & _
    VbCrLf & "          /All: All records (equiv *)" & _
    VbCrLf & "          /r: Name of report file" & _
    VbCrLf & "          /m: Max Nb of records (def = 20000)" & _
    VbCrLf & "          /i: Query interval (in minutes)" & _
    VbCrLf & "          /start: Start date for the query (def = J-7)" & _
    VbCrLf & "          /end: End date (def = J)" & _
    VbCrLf & "          /LastDay: Logs for J-1" & _
    VbCrLf & "          /LastWeek: Logs for last week" & _
    VbCrLf & "          /d: First day of the week" & _
    VbCrLf & "              (0 = Sunday (default), 1 = Monday)" & _
    VbCrLf & "          /LastMonth: Logs for past month, starting the 1st" & _
    VbCrLf & "          /Last30: 30 past days" & _
    VbCrLf & "          /Last24: 24 past hours" & _
    VbCrLf & "          /Last1: Last hour" & _
    VbCrLf & "          /Last7: Last 7 days" & _
    VbCrLf & "          /List: List of available fields in ISA Logs" & _
    VbCrLf & "          /Load: Load wanted fields from a file" & _
    VbCrLf & "          /f: Name of the file used with /List or /Load" & _
    VbCrLf & "          /s: Fields separator (default : ' ; ')" & _
    VbCrLf & _
    VbCrLf & "          /h or /? or /help: Display help"

    WScript.Quit(-1)
End Sub

'----------------------------------------------------------------
'    Purpose : Append a line in a texte file
'             Create file if needed
'    Arguments : Name of the file
'            line to write
'            create or append
'----------------------------------------------------------------
Sub LogAction(strFileLog, strStringToLog, bCreate)
    Dim oFileLog
    Dim TxtLog

    ' Open file
    Set oFileLog = WScript.CreateObject("Scripting.FileSystemObject")

    If bCreate = TRUE Then
        Set TxtLog = oFileLog.OpenTextFile (strFilelog, 2, True)
    Else
        Set TxtLog = oFileLog.OpenTextFile (strFileLog, 8, True)
    End If

    ' Update file with line
    TxtLog.Writeline strStringToLog

    ' Close file
    TxtLog.close
    Set oFileLog = Nothing
 End Sub

'----------------------------------------------------------------
'    Purpose : Validate the first day of the week,
'             then return the corresponding number
'
'    Arguments : String of the day (alhpa or numeric)
'    Return : number for the day, 0 = Sunday
'----------------------------------------------------------------
Function WeekDayNumber(strWeekDay)

    Select Case LCase(strWeekDay)
        Case ""
            WeekDayNumber = -1
        Case "0", "dimanche", "sunday"
            WeekDayNumber = 0
        Case "1", "lundi", "monday"
            WeekDayNumber = 1
        Case "2", "mardi", "tuesday"
            WeekDayNumber = 2
        Case "3", "mercredi", "wednesday"
            WeekDayNumber = 3
        Case "4", "jeudi", "thursday"
            WeekDayNumber = 4
        Case "5", "vendredi", "friday"
            WeekDayNumber = 5
        Case "6", "samedi", "saturday"
            WeekDayNumber = 6
        Case Else
            WeekDayNumber = 0
    End Select

End Function

'----------------------------------------------------------------
'    Purpose : Create list of available fields in ISA logs,
'             then exit the script
'
'    Arguments : Name of file to record list
'----------------------------------------------------------------
Sub ListLogFields(strListFile)
    Dim StrList

    ' ISA Server 2004 full list
    StrList = "Bidirectional" &  VbCrLf & _
        "BytesReceived" & VbCrLf & _
        "BytesSent" & VbCrLf & _
        "CacheInfo" & VbCrLf & _
        "ClientAgent" & VbCrLf & _
        "ClientAuthenticate" & VbCrLf & _
        "ClientIP" & VbCrLf & _
        "ClientUserName" & VbCrLf & _
        "DestHost" & VbCrLf & _
        "DestHostIP" & VbCrLf & _
        "DestHostPort" & VbCrLf & _
        "DestinationNetwork" & VbCrLf & _
        "DestinationProxy" & VbCrLf & _
        "ErrorInfo" & VbCrLf & _
        "FilterInformation" & VbCrLf & _
        "FWAction" & VbCrLf & _
        "Interface" & VbCrLf & _
        "LogDateTime" & VbCrLf & _
        "LogType" & VbCrLf & _
        "MimeType" & VbCrLf & _
        "ObjectSource" & VbCrLf & _
        "Operation" & VbCrLf & _
        "OriginalClientIP" & VbCrLf & _
        "ProcessingTime" & VbCrLf & _
        "Protocol" & VbCrLf & _
        "RawIPHeader" & VbCrLf & _
        "RawPayload" & VbCrLf & _
        "ReferredServer" & VbCrLf & _
        "ResultCode" & VbCrLf & _
        "Rule" & VbCrLf & _
        "ServerName" & VbCrLf & _
        "Service" & VbCrLf & _
        "SourceHost" & VbCrLf & _
        "SourceNetwork" & VbCrLf & _
        "SourceProxy" & VbCrLf & _
        "SrcPort" & VbCrLf & _
        "Transport" & VbCrLf & _
        "URL"
        
    ' Create list file
    call LogAction(strListFile, StrList, TRUE)

    If bDisplay Then
        WScript.Echo "File created:" & VbCrLf & strListFile
    End If

    ' Exit script
    WScript.Quit(0)
End Sub


'----------------------------------------------------------------
'    Purpose: Load list of fields to retrieve from a file
'
'    Arguments: Name of the file
'
'    Global: Fields' array
'----------------------------------------------------------------
Sub LoadFields(StrFileWithFields)
    Dim iCount
    Dim oFileToOpen, txtToOpen

    If bAllRecords Then
        ReDim ArrFields(38)
        
        ' ISA Server 2004 full format
        ArrFields(1) = "Bidirectional"
        ArrFields(2) = "BytesReceived"
        ArrFields(3) = "BytesSent"
        ArrFields(4) = "CacheInfo"
        ArrFields(5) = "ClientAgent"
        ArrFields(6) = "ClientAuthenticate"
        ArrFields(7) = "ClientIP"
        ArrFields(8) = "ClientUserName"
        ArrFields(9) = "DestHost"
        ArrFields(10) = "DestHostIP"
        ArrFields(11) = "DestHostPort"
        ArrFields(12) = "DestinationNetwork"
        ArrFields(13) = "DestinationProxy"
        ArrFields(14) = "ErrorInfo"
        ArrFields(15) = "FilterInformation"
        ArrFields(16) = "FWAction"
        ArrFields(17) = "Interface"
        ArrFields(18) = "LogDateTime"
        ArrFields(19) = "LogType"
        ArrFields(20) = "MimeType"
        ArrFields(21) = "ObjectSource"
        ArrFields(22) = "Operation"
        ArrFields(23) = "OriginalClientIP"
        ArrFields(24) = "ProcessingTime"
        ArrFields(25) = "Protocol"
        ArrFields(26) = "RawIPHeader"
        ArrFields(27) = "RawPayload"
        ArrFields(28) = "ReferredServer"
        ArrFields(29) = "ResultCode"
        ArrFields(30) = "Rule"
        ArrFields(31) = "ServerName"
        ArrFields(32) = "Service"
        ArrFields(33) = "SourceHost"
        ArrFields(34) = "SourceNetwork"
        ArrFields(35) = "SourceProxy"
        ArrFields(36) = "SrcPort"
        ArrFields(37) = "Transport"
        ArrFields(38) = "URL"
    Else
        ' Partial format loaded from file keeping fields' order

        On Error Resume Next
        
        ' open file for reading
        Set oFileToOpen = WScript.CreateObject("Scripting.FileSystemObject")
        Set txtToOpen = oFileToOpen.OpenTextFile(StrFileWithFields, 1, True)

        If Err.Number <> 0 Then
            call ProcessCanceled("List file not found: " & _
                StrFileWithFields)
        End If
        
        On Error Goto 0
        
        iCount = 0
        ' Read lines then store them in array
        Do While Not txtToOpen.AtEndOfStream
            iCount = iCount + 1
            ReDim Preserve ArrFields(iCount)
            ArrFields(iCount) = txtToOpen.ReadLine
        Loop

        ' Close list file
        txtToOpen.close
        Set oFileToOpen = Nothing

    End If
End Sub

'-----------------------------------------------------------------
'    Purpose : Query ISA Logs to retrieve records matching request
'    Arguments : FPC (ISA) Root Object
'                string root URL
'                string Log File Name
'                Max records to retrieve
'                Start Day
'                End Date
'                Max nb of records per query
'
'    Global : Error Code - 0 if ok
'             Error Description
'-----------------------------------------------------------------
Sub SiteLogs(ObjFPC, strUrlRacine, strFileLog, iNbRecords, _
    StartDate, EndDate, ByRef iCounter)
    Dim ObjLogMSDE, ObjFilterExpressions
    Dim LogMSDERecord
    Dim strLogRecord
    Dim strDestinationProxy, strErrorInfo, strInterface, strSourceHost
    Dim strRawIPHeader, strRawPayload, strReferredServer, strSourceProxy
    Dim strSrcPort, strTransport, strURL, strFilterInformation
    Dim iRecordsPerRequest, iLoop
    Dim ArrRecords(20000)
    
    ' Don't set to few nb of records for one query
    If iNbRecords < 10000 Then
        iNbRecords = 10000
    End If

    iRecordsPerRequest = 0
    
    'On Error Resume Next
    
    ' Create log content object
    Set ObjLogMSDE = ObjFPC.GetContainingArray.LogViewer.LogContentMSDE

    ' Create filter object for query
    Set ObjFilterExpressions = CreateObject( "Fpc.FpcFilterExpressions" )

    '-------------------------------------------------------------------------
    ' Set filters
    '-------------------------------------------------------------------------
    ' log type is log viewer
    ObjFilterExpressions.FilterType = FpcLogViewerFilter

    ' List of filter constraints (corresponding to given arguments)
    If bStartDate Then
        ObjFilterExpressions.AddDateFilter FpcFilterByLogTime, FpcOnOrAfter, StartDate
    End If
    If bEndDate Then
        ObjFilterExpressions.AddDateFilter FpcFilterByLogTime, FpcOnOrBefore, EndDate
    End If
    '_________________________________________________________________________
    ' Next options can be used with ISA API, but this script doesn't exploit
    ' them due to the limitation of 20000 records max. bLast* = FALSE)
    ' Workaround is to allways use start and end date splitting the real
    ' search period.

    ' For Last1, 24, 7 , 30, Date() argument is mandatory but not used

    'If bLast1 Then
    '    ObjFilterExpressions.AddDateFilter FpcFilterByLogTime, FpcLastHour, Date()
    'End If
    'If bLast24 Then
    '    ObjFilterExpressions.AddDateFilter FpcFilterByLogTime, FpcLast24Hours, Date()
    'End If
    'If bLast7 Then
    '    ObjFilterExpressions.AddDateFilter FpcFilterByLogTime, FpcLast7days, Date()
    'End If
    'If bLast30 Then
    '    ObjFilterExpressions.AddDateFilter FpcFilterByLogTime, FpcLast30days, Date()
    'End If
    '_________________________________________________________________________
    If strUrlRacine <> "*" Then
        If LCase(Left(strUrlRacine,7)) = "http://" Then
            ObjFilterExpressions.AddStringFilter FpcFilterByUrl, FpcContains, strUrlRacine
        Else
            ObjFilterExpressions.AddStringFilter FpcFilterByRule, FpcContains, strUrlRacine
        End If
    End If

    If Err.Number = 0 Then
        ' Start query
    	ObjLogMSDE.ExecuteQuery ObjFilterExpressions, iNbRecords

	' Load records
	For Each LogMSDERecord In ObjLogMSDE
	    iCounter = iCounter + 1
	    iRecordsPerRequest = iRecordsPerRequest + 1
	    
	    '-----------------------------------------------------------------
	    ' look for fields matching the choosen list (all if full report)
	    '-----------------------------------------------------------------
            strLogRecord = ""
            For iLoop = 1 To UBound(ArrFields)
            
                ' not fields separator if first field !
                If iLoop > 1 Then
                    strLogRecord = strLogRecord & strSeparator
                End If

                Select Case LCase(ArrFields(iLoop))
                    Case "bidirectional"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.Bidirectional)
                    Case "bytesreceived"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.BytesReceived)
                    Case "bytessent"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.BytesSent)
                    Case "cacheinfo"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.CacheInfo)
                    Case "clientagent"
                        strLogRecord = strLogRecord & Replace(LogMSDERecord.ClientAgent, " ", "+")
                    Case "clientauthenticate"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.ClientAuthenticate)
                    Case "clientip"
                        strLogRecord = strLogRecord & LogMSDERecord.ClientIP
                    Case "clientusername"
                        strLogRecord = strLogRecord & LogMSDERecord.ClientUserName
                    Case "desthost"
                        strLogRecord = strLogRecord & LogMSDERecord.DestHost
                    Case "desthostip"
                        strLogRecord = strLogRecord & LogMSDERecord.DestHostIP
                    Case "desthostport"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.DestHostPort)
                    Case "destinationnetwork"
                        strLogRecord = strLogRecord & LogMSDERecord.DestinationNetwork
                    Case "destinationproxy"
                        strDestinationProxy = Replace(LogMSDERecord.DestinationProxy, " ", "-")
                        If strDestinationProxy = "" Then
                            strDestinationProxy = "-"
                        End If
                        strLogRecord = strLogRecord & strDestinationProxy
                    Case "errorinfo"
                        strErrorInfo = Replace(CStr(LogMSDERecord.ErrorInfo), " ", "-")
                        If strErrorInfo = "" Then
                            strErrorInfo = "-"
                        End If
                        strLogRecord = strLogRecord & strErrorInfo
                    Case "filterinformation"
                        strFilterInformation = Replace(CStr(LogMSDERecord.FilterInformation), " ", "-")
                        If strFilterInformation = "" Then
                            strFilterInformation = "-"
                        End If
                        strLogRecord = strLogRecord & strFilterInformation
                    Case "fwaction"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.FWAction)
                    Case "interface"
                        strInterface = Replace(LogMSDERecord.Interface, " ", "-")
                        If strInterface = "" Then
                            strInterface = "-"
                        End If
                        strLogRecord = strLogRecord & strInterface
                    Case "logdatetime"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.LogTime)
                    Case "logtype"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.LogType)
                    Case "mimetype"
                        strLogRecord = strLogRecord & LogMSDERecord.MimeType
                    Case "objectsource"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.ObjectSource)
                    Case "operation"
                        strLogRecord = strLogRecord & LogMSDERecord.Operation
                    Case "originalclientip"
                        strLogRecord = strLogRecord & LogMSDERecord.OriginalClientIP
                    Case "processingtime"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.ProcessingTime)
                    Case "protocol"
                        strLogRecord = strLogRecord & LogMSDERecord.Protocol
                    Case "rawipheader"
                        strRawIPHeader = Replace(LogMSDERecord.RawIPHeader, " ", "-")
                        If strRawIPHeader = "" Then
                            strRawIPHeader = "-"
                        End If
                        strLogRecord = strLogRecord & strRawIPHeader
                    Case "rawpayload"
                        strRawPayload = Replace(LogMSDERecord.RawPayload, " ", "-")
                        If strRawPayload = "" Then
                            strRawPayload = "-"
                        End If
                        strLogRecord = strLogRecord & strRawPayload
                    Case "referredserver"
                        strReferredServer = Replace(LogMSDERecord.ReferredServer, " ", "-")
                        If strReferredServer = "" Then
                            strReferredServer = "-"
                        End If
                        strLogRecord = strLogRecord & strReferredServer
                    Case "resultcode"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.ResultCode)
                    Case "rule"
                        strLogRecord = strLogRecord & Replace(LogMSDERecord.Rule, " ", "+")
                    Case "servername"
                        strLogRecord = strLogRecord & LogMSDERecord.ServerName
                    Case "service"
                        strLogRecord = strLogRecord & CStr(LogMSDERecord.Service)
                    Case "sourcehost"
                        strSourceHost = Replace(LogMSDERecord.SourceHost, " ", "-")
                        If strSourceHost = "" Then
                            strSourceHost = "-"
                        End If
                        strLogRecord = strLogRecord & strSourceHost
                    Case "sourcenetwork"
                        strLogRecord = strLogRecord & Replace(LogMSDERecord.SourceNetwork, " ", "+")
                    Case "sourceproxy"
                        strSourceProxy = Replace(LogMSDERecord.SourceProxy, " ", "-")
                        If strSourceProxy = "" Then
                            strSourceProxy = "-"
                        End If
                        strLogRecord = strLogRecord & strSourceProxy
                    Case "srcport"
                        strSrcPort = Replace(CStr(LogMSDERecord.SrcPort), " ", "-")
                        If strSrcPort = "" Then
                            strSrcPort = "-"
                        End If
                        strLogRecord = strLogRecord & strSrcPort
                    Case "transport"
                        strTransport = Replace(LogMSDERecord.Transport, " ", "-")
                        If strTransport = "" Then
                            strTransport = "-"
                        End If
                        strLogRecord = strLogRecord & strTransport
                    Case "url"
                        strURL = Replace(LogMSDERecord.URL, " ", "-")
                        If strURL = "" Then
                            strURL = "-"
                        End If
                        strLogRecord = strLogRecord & strURL
                End Select

            Next

            ' Temporary storage of the record (due to default reverse order)
            ArrRecords(iRecordsPerRequest) = strLogRecord
        Next
        
        '---------------------------------------------------------------------
        ' Reverse tri order (newer first)
        '---------------------------------------------------------------------
        For iLoop = iRecordsPerRequest To 1 Step -1
            call LogAction(strFileLog, ArrRecords(iLoop), FALSE)
        Next

    End If

    ' Record an alert if the max nb of records was reached
    If iRecordsPerRequest >= iNbRecords Then
        strLogRecord = "    ----> Warning, max nb of records was reached." & _
            VbCrLf & "          Reduce the query interval (option /i)"
        call LogAction(strFileLog, strLogRecord, FALSE)
    End If

    ObjLogMSDE.EndQuery
    Set ObjFilterExpressions = Nothing
End Sub

'===========================================================================
' Set constants (Globals)
'===========================================================================

    ' FpcFilterType
    Const FpcNoFilterType      = 0
    Const FpcSessionsFilter    = 1
    Const FpcLogViewerFilter   = 2

    ' FpcFilterCondition
    Const FpcNoCondition       = &h00000000
    Const FpcEqual             = &h00000001
    Const FpcNotEqual          = &h00000002
    Const FpcGreaterOrEqual    = &h00000004
    Const FpcLessOrEqual       = &h00000008
    Const FpcOnOrAfter         = &h00000010
    Const FpcOnOrBefore        = &h00000020
    Const FpcContains          = &h00000040
    Const FpcLiveData          = &h00000080
    Const FpcLastHour          = &h00000100
    Const FpcLast24Hours       = &h00000200
    Const FpcLast7days         = &h00000400
    Const FpcLast30days        = &h00000800
    Const FpcNotContains       = &h00001000

    ' FpcFilterCriteria
    Const FpcNoFilter                   = 0
    Const FpcFilterByClientIP           = 1
    Const FpcFilterByClientUserName     = 2
    Const FpcFilterByClientAgent        = 3
    Const FpcFilterByAuthenticated      = 4
    Const FpcFilterByService            = 5
    Const FpcFilterByServername         = 6
    Const FpcFilterByReferringServer    = 7
    Const FpcFilterByDestHost           = 8
    Const FpcFilterByDestHostIP         = 9
    Const FpcFilterByProtocol           = 10
    Const FpcFilterByTransport          = 11
    Const FpcFilterByOperation          = 12
    Const FpcFilterByUrl                = 13
    Const FpcFilterByRule               = 14
    Const FpcFilterByFilterInfo         = 15
    Const FpcFilterByMimetype           = 16
    Const FpcFilterByObjectSource       = 17
    Const FpcFilterByLogTime            = 18
    Const FpcFilterByCacheInfo          = 19
    Const FpcFilterByErrorInfo          = 20
    Const FpcFilterByDestHostPort       = 21
    Const FpcFilterBySrcPort            = 22
    Const FpcFilterByLogType            = 23
    Const FpcFilterBySessionType        = 24
    Const FpcFilterByActivation         = 25
    Const FpcFilterByClientComputerName = 26
    Const FpcFilterBySourceNetwork      = 27
    Const FpcFilterByDestinationNetwork = 28
    Const FpcFilterBySourceProxy        = 29
    Const FpcFilterByDestinationProxy   = 30
    Const FpcFilterByAction             = 31
    Const FpcFilterByBidirectional      = 32
    Const FpcFilterByInterface          = 33
    Const FpcFilterByHeader             = 34
    Const FpcFilterByPayload            = 35
    Const FpcFilterProcessingTime       = 36
    Const FpcFilterByResultCode         = 37
    Const FpcFilterByBytesSent          = 38
    Const FpcFilterByBytesReceived      = 39
    Const FpcFilterByApplicationName    = 40
    Const FpcFilterByOriginalClientIP   = 41
    Const FpcFilterByHTTPCode           = 42

    ' FpcAction
    Const FpcActionNotLogged              = 0
    Const FpcActionBind                   = 1
    Const FpcActionListen                 = 2
    Const FpcActionGHBN                   = 3
    Const FpcActionGHBA                   = 4
    Const FpcActionRedirectBind           = 5
    Const FpcActionEstablish              = 6
    Const FpcActionTerminate              = 7
    Const FpcActionDenied                 = 8
    Const FpcActionAllowed                = 9
    Const FpcActionFailed                 = 10
    Const FpcActionIntermediate           = 11
    Const FpcActionSuccessfulConnection   = 12
    Const FpcActionUnsuccessfulConnection = 13
    Const FpcActionDisconnection          = 14
    Const FpcActionUserclearedQuarantine  = 15
    Const FpcActionQuarantinetimeout      = 16

